/**
 * @description Test class for QuikFormsSmartValidationPlugin. Covers plugin
 * info, initialization, email validation, phone validation, empty value
 * pass-through, non-email field bypass, validation mode, and async delegation.
 *
 * @author QuikForms
 * @since 1.0
 */
@isTest
private class QuikFormsSmartValidationPluginTest {

    // -------------------------------------------------------------------------
    // Helper: build a validation context
    // -------------------------------------------------------------------------

    private static QuikFormsValidationContext buildContext(
        String fieldId,
        Object fieldValue,
        String fieldType,
        String objectField
    ) {
        QuikFormsValidationContext ctx = new QuikFormsValidationContext();
        ctx.fieldId = fieldId;
        ctx.fieldValue = fieldValue;
        ctx.fieldType = fieldType;
        ctx.objectField = objectField;
        ctx.allFieldValues = new Map<String, Object>{ fieldId => fieldValue };
        ctx.pluginConfiguration = new Map<String, Object>();
        return ctx;
    }

    // -------------------------------------------------------------------------
    // Tests: IQuikFormsPlugin
    // -------------------------------------------------------------------------

    @isTest
    static void testGetPluginInfo() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();

        Test.startTest();
        QuikFormsPluginInfo info = plugin.getPluginInfo();
        Test.stopTest();

        System.assertEquals('Smart Validation Suite', info.name, 'Plugin name should match');
        System.assertEquals('1.0.0', info.version, 'Plugin version should match');
        System.assert(
            info.description.contains('Server-side validation'),
            'Description should mention server-side validation'
        );
        System.assertEquals(false, info.supportsCallouts, 'Should not support callouts');
        System.assertEquals(1, info.supportedHooks.size(), 'Should support exactly one hook');
        System.assertEquals('onValidate', info.supportedHooks[0], 'Supported hook should be onValidate');
    }

    @isTest
    static void testInitializeAndIsReady() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();

        System.assertEquals(false, plugin.isReady(), 'Should not be ready before initialization');

        Test.startTest();
        plugin.initialize(new Map<String, Object>{ 'testKey' => 'testValue' });
        Test.stopTest();

        System.assertEquals(true, plugin.isReady(), 'Should be ready after initialization');
    }

    @isTest
    static void testInitializeWithNull() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();

        Test.startTest();
        plugin.initialize(null);
        Test.stopTest();

        System.assertEquals(true, plugin.isReady(), 'Should be ready even with null config');
    }

    // -------------------------------------------------------------------------
    // Tests: Email validation
    // -------------------------------------------------------------------------

    @isTest
    static void testValidateEmailValid() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'email1', 'user@example.com', 'inputText', 'Email'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Valid email should pass validation');
        System.assertEquals(0, result.errorMessages.size(), 'Should have no error messages');
    }

    @isTest
    static void testValidateEmailValidComplex() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'email1', 'first.last+tag@sub.domain.co.uk', 'inputText', 'Email'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Complex valid email should pass validation');
    }

    @isTest
    static void testValidateEmailInvalid() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'email1', 'not-an-email', 'inputText', 'Email'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(false, result.isValid, 'Invalid email should fail validation');
        System.assertEquals(1, result.errorMessages.size(), 'Should have one error message');
        System.assert(
            result.errorMessages[0].contains('valid email'),
            'Error message should mention valid email'
        );
    }

    @isTest
    static void testValidateEmailInvalidMissingDomain() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'email1', 'user@', 'inputText', 'ContactEmail'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(false, result.isValid, 'Email without domain should fail');
        System.assert(result.errorMessages.size() > 0, 'Should have error messages');
    }

    @isTest
    static void testValidateEmailDetectedByFieldType() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'field1', 'bad-email', 'inputEmail', 'SomeField__c'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(false, result.isValid, 'Should detect email field by fieldType');
    }

    // -------------------------------------------------------------------------
    // Tests: Phone validation
    // -------------------------------------------------------------------------

    @isTest
    static void testValidatePhoneValid() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'phone1', '(555) 123-4567', 'inputText', 'Phone'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Valid phone should pass validation');
        System.assertEquals(0, result.errorMessages.size(), 'Should have no error messages');
    }

    @isTest
    static void testValidatePhoneValidInternational() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'phone1', '+1-555-123-4567', 'inputText', 'Phone'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'International phone format should pass');
    }

    @isTest
    static void testValidatePhoneInvalid() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'phone1', '555-1234', 'inputText', 'Phone'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(false, result.isValid, 'Phone with fewer than 10 digits should fail');
        System.assertEquals(1, result.errorMessages.size(), 'Should have one error message');
        System.assert(
            result.errorMessages[0].contains('10 digits'),
            'Error message should mention 10 digits'
        );
    }

    @isTest
    static void testValidatePhoneDetectedByFieldType() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'field1', '12345', 'inputPhone', 'SomeField__c'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(false, result.isValid, 'Should detect phone field by fieldType');
    }

    // -------------------------------------------------------------------------
    // Tests: Empty/null values
    // -------------------------------------------------------------------------

    @isTest
    static void testValidateEmptyValue() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        // Test with empty string
        QuikFormsValidationContext ctx = buildContext(
            'email1', '', 'inputText', 'Email'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Empty value should pass (required check is not this plugin\'s job)');
    }

    @isTest
    static void testValidateNullValue() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'email1', null, 'inputText', 'Email'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Null value should pass validation');
    }

    @isTest
    static void testValidateNullContext() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(null);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Null context should pass validation');
    }

    @isTest
    static void testValidateBlankValue() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'phone1', '   ', 'inputText', 'Phone'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Blank/whitespace value should pass');
    }

    // -------------------------------------------------------------------------
    // Tests: Non-email/phone fields
    // -------------------------------------------------------------------------

    @isTest
    static void testValidateNonEmailField() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'name1', 'John Doe', 'inputText', 'Name'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Non-email, non-phone field should pass through');
        System.assertEquals(0, result.errorMessages.size(), 'Should have no error messages');
    }

    @isTest
    static void testValidateGenericTextField() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'description1', 'Some text value', 'inputTextarea', 'Description__c'
        );

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Generic text field should pass through');
    }

    // -------------------------------------------------------------------------
    // Tests: Validation mode
    // -------------------------------------------------------------------------

    @isTest
    static void testGetValidationMode() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();

        Test.startTest();
        QuikFormsValidationMode mode = plugin.getValidationMode();
        Test.stopTest();

        System.assertEquals(
            QuikFormsValidationMode.SYNC_ONLY,
            mode,
            'Validation mode should be SYNC_ONLY'
        );
    }

    // -------------------------------------------------------------------------
    // Tests: Async delegation
    // -------------------------------------------------------------------------

    @isTest
    static void testValidateAsync() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'email1', 'invalid-email', 'inputText', 'Email'
        );

        Test.startTest();
        QuikFormsValidationResult asyncResult = plugin.validateAsync(ctx);
        QuikFormsValidationResult syncResult = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(
            syncResult.isValid,
            asyncResult.isValid,
            'Async should return same result as sync'
        );
        System.assertEquals(
            syncResult.errorMessages.size(),
            asyncResult.errorMessages.size(),
            'Async should have same number of error messages as sync'
        );
    }

    // -------------------------------------------------------------------------
    // Tests: Custom regex rules
    // -------------------------------------------------------------------------

    @isTest
    static void testCustomRegexRulePass() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'zipcode1', '90210', 'inputText', 'PostalCode__c'
        );
        ctx.pluginConfiguration = new Map<String, Object>{
            'customRegexRules' => new List<Object>{
                new Map<String, Object>{
                    'pattern' => '^\\d{5}$',
                    'message' => 'Must be a 5-digit ZIP code.'
                }
            }
        };

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Value matching regex should pass');
    }

    @isTest
    static void testCustomRegexRuleFail() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'zipcode1', 'ABCDE', 'inputText', 'PostalCode__c'
        );
        ctx.pluginConfiguration = new Map<String, Object>{
            'customRegexRules' => new List<Object>{
                new Map<String, Object>{
                    'pattern' => '^\\d{5}$',
                    'message' => 'Must be a 5-digit ZIP code.'
                }
            }
        };

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(false, result.isValid, 'Value not matching regex should fail');
        System.assert(
            result.errorMessages[0].contains('5-digit ZIP'),
            'Error message should contain custom message'
        );
    }

    @isTest
    static void testCustomRegexRuleTargetedField() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        // Rule targets a different field, should not apply
        QuikFormsValidationContext ctx = buildContext(
            'name1', 'ABCDE', 'inputText', 'Name__c'
        );
        ctx.pluginConfiguration = new Map<String, Object>{
            'customRegexRules' => new List<Object>{
                new Map<String, Object>{
                    'field' => 'zipcode1',
                    'pattern' => '^\\d{5}$',
                    'message' => 'Must be a 5-digit ZIP code.'
                }
            }
        };

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(true, result.isValid, 'Rule targeting a different field should not apply');
    }

    // -------------------------------------------------------------------------
    // Tests: Plugin config fieldType override
    // -------------------------------------------------------------------------

    @isTest
    static void testFieldTypeOverrideEmail() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        // objectField does not contain 'email', but config says it is an email field
        QuikFormsValidationContext ctx = buildContext(
            'custom1', 'not-an-email', 'inputText', 'CustomField__c'
        );
        ctx.pluginConfiguration = new Map<String, Object>{
            'fieldType' => 'email'
        };

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(false, result.isValid, 'Should validate as email when config overrides type');
    }

    @isTest
    static void testFieldTypeOverridePhone() {
        QuikFormsSmartValidationPlugin plugin = new QuikFormsSmartValidationPlugin();
        plugin.initialize(new Map<String, Object>());

        QuikFormsValidationContext ctx = buildContext(
            'custom1', '12345', 'inputText', 'CustomField__c'
        );
        ctx.pluginConfiguration = new Map<String, Object>{
            'fieldType' => 'phone'
        };

        Test.startTest();
        QuikFormsValidationResult result = plugin.validateSync(ctx);
        Test.stopTest();

        System.assertEquals(false, result.isValid, 'Should validate as phone when config overrides type');
    }
}
