/**
 * @description Server-side validation plugin for QuikForms. Implements both
 * IQuikFormsFieldValidator and IQuikFormsPlugin to provide robust email format
 * validation, phone number pattern checking, and configurable custom regex
 * rules that are enforced on the server regardless of client-side state.
 *
 * This class complements the client-side JavaScript plugin by providing a
 * security backstop: even if JavaScript validation is bypassed, these checks
 * run during form submission processing on the server.
 *
 * @author QuikForms
 * @since 1.0
 */
global class QuikFormsSmartValidationPlugin implements IQuikFormsFieldValidator, IQuikFormsPlugin {

    // -------------------------------------------------------------------------
    // Constants
    // -------------------------------------------------------------------------

    @TestVisible
    private static final String EMAIL_REGEX =
        '^[a-zA-Z0-9.!#$%&\'*+/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$';

    @TestVisible
    private static final String PHONE_REGEX = '\\d';

    @TestVisible
    private static final Integer PHONE_MIN_DIGITS = 10;

    private static final String ERROR_INVALID_EMAIL = 'Please enter a valid email address.';
    private static final String ERROR_INVALID_PHONE = 'Phone number must contain at least {0} digits.';

    // -------------------------------------------------------------------------
    // Instance state
    // -------------------------------------------------------------------------

    private Map<String, Object> config;
    private Boolean initialized = false;

    // -------------------------------------------------------------------------
    // IQuikFormsPlugin methods
    // -------------------------------------------------------------------------

    /**
     * @description Returns metadata about this plugin.
     * @return QuikFormsPluginInfo containing name, version, and capabilities
     */
    global QuikFormsPluginInfo getPluginInfo() {
        QuikFormsPluginInfo info = new QuikFormsPluginInfo();
        info.name = 'Smart Validation Suite';
        info.version = '1.0.0';
        info.description = 'Server-side validation for email format, phone patterns, and custom regex rules';
        info.supportsCallouts = false;
        info.supportedHooks = new List<String>{ 'onValidate' };
        return info;
    }

    /**
     * @description Initializes the plugin with the provided configuration.
     * @param configuration Map of configuration options
     */
    global void initialize(Map<String, Object> configuration) {
        this.config = configuration != null ? configuration : new Map<String, Object>();
        this.initialized = true;
    }

    /**
     * @description Returns whether the plugin has been initialized and is ready.
     * @return True if initialized
     */
    global Boolean isReady() {
        return this.initialized;
    }

    // -------------------------------------------------------------------------
    // IQuikFormsFieldValidator methods
    // -------------------------------------------------------------------------

    /**
     * @description Synchronous field validation. Detects field type from the
     * context and applies the appropriate validation rules:
     *   1. Email fields: validated against RFC 5322 regex
     *   2. Phone fields: must contain at least 10 digits
     *   3. Custom regex rules defined in pluginConfiguration
     *
     * Empty/null values pass validation -- enforcing "required" is not this
     * validator's responsibility.
     *
     * @param context QuikFormsValidationContext with field metadata and value
     * @return QuikFormsValidationResult success or failure with messages
     */
    global QuikFormsValidationResult validateSync(QuikFormsValidationContext context) {
        // Null or empty values always pass -- required-ness is handled elsewhere
        if (context == null || context.fieldValue == null) {
            return QuikFormsValidationResult.success();
        }

        String value = String.valueOf(context.fieldValue).trim();
        if (String.isBlank(value)) {
            return QuikFormsValidationResult.success();
        }

        QuikFormsValidationResult result = new QuikFormsValidationResult();

        // Determine field type from context
        Boolean isEmail = isEmailField(context);
        Boolean isPhone = isPhoneField(context);

        // 1. Email validation
        if (isEmail) {
            if (!isValidEmail(value)) {
                result.addError(ERROR_INVALID_EMAIL);
            }
        }

        // 2. Phone validation
        if (isPhone) {
            if (!isValidPhone(value)) {
                String msg = ERROR_INVALID_PHONE.replace('{0}', String.valueOf(PHONE_MIN_DIGITS));
                result.addError(msg);
            }
        }

        // 3. Custom regex rules from configuration
        applyCustomRegexRules(context, value, result);

        return result;
    }

    /**
     * @description Asynchronous validation. Delegates to synchronous validation
     * since this plugin does not require callouts or async processing.
     *
     * @param context QuikFormsValidationContext
     * @return QuikFormsValidationResult
     */
    global QuikFormsValidationResult validateAsync(QuikFormsValidationContext context) {
        return validateSync(context);
    }

    /**
     * @description Returns the validation mode. This plugin uses sync-only
     * validation since no external callouts are needed.
     *
     * @return QuikFormsValidationMode.SYNC_ONLY
     */
    global QuikFormsValidationMode getValidationMode() {
        return QuikFormsValidationMode.SYNC_ONLY;
    }

    // -------------------------------------------------------------------------
    // Private helpers
    // -------------------------------------------------------------------------

    /**
     * @description Check whether the field should be treated as an email field
     * based on objectField name, fieldType, or explicit configuration.
     */
    @TestVisible
    private Boolean isEmailField(QuikFormsValidationContext context) {
        // Check explicit fieldType configuration
        if (context.pluginConfiguration != null &&
            context.pluginConfiguration.containsKey('fieldType')) {
            String configuredType = String.valueOf(context.pluginConfiguration.get('fieldType'));
            if (configuredType != null && configuredType.equalsIgnoreCase('email')) {
                return true;
            }
        }

        // Check objectField name
        if (String.isNotBlank(context.objectField) &&
            context.objectField.toLowerCase().contains('email')) {
            return true;
        }

        // Check fieldType
        if (String.isNotBlank(context.fieldType)) {
            String ft = context.fieldType.toLowerCase();
            if (ft == 'inputemail' || ft == 'email') {
                return true;
            }
        }

        return false;
    }

    /**
     * @description Check whether the field should be treated as a phone field
     * based on objectField name, fieldType, or explicit configuration.
     */
    @TestVisible
    private Boolean isPhoneField(QuikFormsValidationContext context) {
        // Check explicit fieldType configuration
        if (context.pluginConfiguration != null &&
            context.pluginConfiguration.containsKey('fieldType')) {
            String configuredType = String.valueOf(context.pluginConfiguration.get('fieldType'));
            if (configuredType != null && configuredType.equalsIgnoreCase('phone')) {
                return true;
            }
        }

        // Check objectField name
        if (String.isNotBlank(context.objectField) &&
            context.objectField.toLowerCase().contains('phone')) {
            return true;
        }

        // Check fieldType
        if (String.isNotBlank(context.fieldType)) {
            String ft = context.fieldType.toLowerCase();
            if (ft == 'inputphone' || ft == 'phone' || ft == 'tel') {
                return true;
            }
        }

        return false;
    }

    /**
     * @description Validate an email address against the RFC 5322 regex.
     */
    @TestVisible
    private Boolean isValidEmail(String value) {
        if (String.isBlank(value)) return false;

        Pattern emailPattern = Pattern.compile(EMAIL_REGEX);
        Matcher emailMatcher = emailPattern.matcher(value);
        return emailMatcher.matches();
    }

    /**
     * @description Validate that a phone number contains the minimum number
     * of digits, regardless of formatting characters.
     */
    @TestVisible
    private Boolean isValidPhone(String value) {
        if (String.isBlank(value)) return false;

        // Count digits in the value
        String digitsOnly = value.replaceAll('[^0-9]', '');
        return digitsOnly.length() >= PHONE_MIN_DIGITS;
    }

    /**
     * @description Apply any custom regex validation rules from the plugin
     * configuration. Rules are expected in the config as a list under the key
     * 'customRegexRules', where each rule has 'pattern' and 'message' keys.
     *
     * Example config:
     * {
     *   "customRegexRules": [
     *     { "pattern": "^[A-Z]{2}\\d{6}$", "message": "Must be 2 letters followed by 6 digits" }
     *   ]
     * }
     */
    @TestVisible
    private void applyCustomRegexRules(QuikFormsValidationContext context, String value, QuikFormsValidationResult result) {
        Map<String, Object> pluginConfig = context.pluginConfiguration;
        if (pluginConfig == null || !pluginConfig.containsKey('customRegexRules')) {
            // Also check instance-level config
            if (this.config == null || !this.config.containsKey('customRegexRules')) {
                return;
            }
            pluginConfig = this.config;
        }

        Object rulesObj = pluginConfig.get('customRegexRules');
        if (!(rulesObj instanceof List<Object>)) {
            return;
        }

        List<Object> rules = (List<Object>) rulesObj;
        for (Object ruleObj : rules) {
            if (!(ruleObj instanceof Map<String, Object>)) continue;

            Map<String, Object> rule = (Map<String, Object>) ruleObj;
            String patternStr = rule.containsKey('pattern') ? String.valueOf(rule.get('pattern')) : null;
            String message = rule.containsKey('message') ? String.valueOf(rule.get('message')) : 'Value does not match the required pattern.';

            // Check if this rule targets a specific field
            String targetField = rule.containsKey('field') ? String.valueOf(rule.get('field')) : null;
            if (String.isNotBlank(targetField) && !targetField.equalsIgnoreCase(context.fieldId) &&
                !targetField.equalsIgnoreCase(context.objectField)) {
                continue; // Rule is for a different field
            }

            if (String.isNotBlank(patternStr)) {
                try {
                    Pattern customPattern = Pattern.compile(patternStr);
                    Matcher customMatcher = customPattern.matcher(value);
                    if (!customMatcher.matches()) {
                        result.addError(message);
                    }
                } catch (Exception e) {
                    // Invalid regex in configuration - log warning but don't block
                    result.addWarning('Validation rule has invalid pattern: ' + patternStr);
                }
            }
        }
    }
}
