// QuikForms Input Masking Plugin v1.0.0
// Applies format masks to text fields as users type
(function() {
  'use strict';

  // ==================== Predefined Mask Patterns ====================

  var PREDEFINED_MASKS = {
    'us-phone':        '(###) ###-####',
    'ssn':             '###-##-####',
    'ein':             '##-#######',
    'credit-card':     '#### #### #### ####',
    'us-zip':          '#####',
    'us-zip-plus4':    '#####-####',
    'ca-postal':       'A#A #A#',
    'date':            '##/##/####',
    'date-iso':        '####-##-##',
    'time':            '##:##',
    'time-seconds':    '##:##:##',
    'currency-us':     '$#,###.##',
    'percentage':      '###.##%'
  };

  // ==================== Mask Character Definitions ====================

  // # = digit (0-9)
  // A = letter (a-zA-Z)
  // * = alphanumeric (a-zA-Z0-9)
  // All other characters are treated as literals

  var MASK_CHARS = {
    '#': /[0-9]/,
    'A': /[a-zA-Z]/,
    '*': /[a-zA-Z0-9]/
  };

  // ==================== CSS Variables ====================

  var CSS_INJECTED = false;
  var CSS_RULES = [
    '.qf-masked-field {',
    '  font-family: var(--qf-mask-font-family, "SF Mono", "Consolas", "Courier New", monospace);',
    '  letter-spacing: var(--qf-mask-letter-spacing, 0.5px);',
    '}',
    '.qf-masked-field::placeholder {',
    '  color: var(--qf-mask-placeholder-color, #999);',
    '  opacity: var(--qf-mask-placeholder-opacity, 0.7);',
    '}',
    '.qf-mask-incomplete {',
    '  border-color: var(--qf-mask-incomplete-border, #e67e22) !important;',
    '}',
    '.qf-mask-error {',
    '  border-color: var(--qf-mask-error-border, #e74c3c) !important;',
    '}'
  ].join('\n');

  function injectCSS() {
    if (CSS_INJECTED) {
      return;
    }
    var style = document.createElement('style');
    style.setAttribute('data-qf-plugin', 'input-masking');
    style.textContent = CSS_RULES;
    document.head.appendChild(style);
    CSS_INJECTED = true;
  }

  // ==================== Plugin State ====================

  // Map of fieldId -> mask configuration
  var fieldMasks = {};

  // Map of fieldId -> attached listener references (for cleanup)
  var fieldListeners = {};

  // Global configuration reference
  var pluginConfig = null;

  // ==================== Utility Functions ====================

  /**
   * Resolve a mask pattern string. If it matches a predefined mask name,
   * return the predefined pattern; otherwise return the string as-is.
   * @param {string} mask - Mask pattern or predefined name
   * @returns {string} Resolved mask pattern
   */
  function resolveMask(mask) {
    if (!mask) {
      return '';
    }
    var lower = mask.toLowerCase().replace(/\s+/g, '-');
    if (PREDEFINED_MASKS[lower]) {
      return PREDEFINED_MASKS[lower];
    }
    return mask;
  }

  /**
   * Test whether a character matches a mask token at a given position.
   * @param {string} char - Single character to test
   * @param {string} maskChar - Mask character at that position
   * @returns {boolean}
   */
  function charMatchesMask(char, maskChar) {
    var pattern = MASK_CHARS[maskChar];
    if (pattern) {
      return pattern.test(char);
    }
    // Literal: character must match exactly
    return char === maskChar;
  }

  /**
   * Check if a mask character is a placeholder (expects user input)
   * rather than a literal.
   * @param {string} maskChar
   * @returns {boolean}
   */
  function isPlaceholder(maskChar) {
    return MASK_CHARS.hasOwnProperty(maskChar);
  }

  /**
   * Build a display placeholder from a mask pattern.
   * Replaces # with 0, A with _, * with _ and keeps literals.
   * @param {string} mask
   * @returns {string}
   */
  function buildPlaceholder(mask) {
    var result = '';
    for (var i = 0; i < mask.length; i++) {
      var ch = mask[i];
      if (ch === '#') {
        result += '0';
      } else if (ch === 'A') {
        result += '_';
      } else if (ch === '*') {
        result += '_';
      } else {
        result += ch;
      }
    }
    return result;
  }

  /**
   * Apply a mask pattern to a raw (unformatted) value.
   * Returns the formatted string.
   * @param {string} rawValue - Unformatted input characters
   * @param {string} mask - Mask pattern
   * @returns {string} Formatted value
   */
  function applyMask(rawValue, mask) {
    if (!mask || !rawValue) {
      return rawValue || '';
    }

    var result = '';
    var rawIndex = 0;

    for (var maskIndex = 0; maskIndex < mask.length && rawIndex < rawValue.length; maskIndex++) {
      var maskChar = mask[maskIndex];

      if (isPlaceholder(maskChar)) {
        // Find the next raw character that matches this placeholder
        while (rawIndex < rawValue.length) {
          var inputChar = rawValue[rawIndex];
          rawIndex++;

          if (MASK_CHARS[maskChar].test(inputChar)) {
            // For letter masks, preserve case from input
            result += inputChar;
            break;
          }
          // Skip characters that don't match the expected type
        }
      } else {
        // Literal character in the mask - insert it automatically
        result += maskChar;
        // If the raw value has this literal at the current position, consume it
        if (rawIndex < rawValue.length && rawValue[rawIndex] === maskChar) {
          rawIndex++;
        }
      }
    }

    return result;
  }

  /**
   * Strip formatting from a masked value, returning only the user-entered
   * characters (no literals from the mask).
   * @param {string} formattedValue - The displayed (formatted) value
   * @param {string} mask - Mask pattern
   * @returns {string} Raw value with only user-entered characters
   */
  function stripMask(formattedValue, mask) {
    if (!mask || !formattedValue) {
      return formattedValue || '';
    }

    var result = '';
    for (var i = 0; i < formattedValue.length && i < mask.length; i++) {
      if (isPlaceholder(mask[i])) {
        result += formattedValue[i];
      }
    }
    return result;
  }

  /**
   * Extract only the characters the user typed (digits, letters, alphanumerics)
   * from a string, ignoring any literal/formatting characters.
   * @param {string} value
   * @returns {string}
   */
  function extractRawInput(value) {
    if (!value) {
      return '';
    }
    return value.replace(/[^a-zA-Z0-9]/g, '');
  }

  /**
   * Count how many placeholder positions exist in a mask pattern.
   * @param {string} mask
   * @returns {number}
   */
  function countPlaceholders(mask) {
    var count = 0;
    for (var i = 0; i < mask.length; i++) {
      if (isPlaceholder(mask[i])) {
        count++;
      }
    }
    return count;
  }

  /**
   * Check whether a formatted value is complete (all placeholder positions filled).
   * @param {string} formattedValue
   * @param {string} mask
   * @returns {boolean}
   */
  function isMaskComplete(formattedValue, mask) {
    if (!formattedValue || !mask) {
      return false;
    }
    if (formattedValue.length < mask.length) {
      return false;
    }
    for (var i = 0; i < mask.length; i++) {
      if (isPlaceholder(mask[i])) {
        if (i >= formattedValue.length || !MASK_CHARS[mask[i]].test(formattedValue[i])) {
          return false;
        }
      }
    }
    return true;
  }

  // ==================== Field Interaction Handlers ====================

  /**
   * Handle keydown events to control character entry.
   * Prevents invalid characters at each mask position.
   * @param {KeyboardEvent} event
   */
  function handleKeyDown(event) {
    var input = event.target;
    var fieldId = input.id.replace(/^field-/, '');
    var maskInfo = fieldMasks[fieldId];

    if (!maskInfo) {
      return;
    }

    var mask = maskInfo.pattern;
    var key = event.key;

    // Allow control keys
    if (event.ctrlKey || event.metaKey || event.altKey) {
      return;
    }

    // Allow navigation and editing keys
    var allowedKeys = [
      'Backspace', 'Delete', 'Tab', 'Escape', 'Enter',
      'ArrowLeft', 'ArrowRight', 'ArrowUp', 'ArrowDown',
      'Home', 'End'
    ];
    if (allowedKeys.indexOf(key) !== -1) {
      return;
    }

    // For printable characters, check if they are valid at the current position
    if (key.length === 1) {
      var cursorPos = input.selectionStart;
      var currentValue = input.value;

      // If there is a selection, allow the keystroke (it will replace selected text)
      if (input.selectionStart !== input.selectionEnd) {
        return;
      }

      // If we have reached the end of the mask, prevent further input
      if (cursorPos >= mask.length) {
        event.preventDefault();
        return;
      }

      // Find the next placeholder position from the cursor
      var targetPos = cursorPos;
      while (targetPos < mask.length && !isPlaceholder(mask[targetPos])) {
        targetPos++;
      }

      // If no more placeholder positions, prevent input
      if (targetPos >= mask.length) {
        event.preventDefault();
        return;
      }

      // Validate the character against the mask at the target position
      var maskChar = mask[targetPos];
      if (!MASK_CHARS[maskChar].test(key)) {
        event.preventDefault();
        return;
      }
    }
  }

  /**
   * Handle input events to apply the mask after each keystroke.
   * @param {InputEvent} event
   */
  function handleInput(event) {
    var input = event.target;
    var fieldId = input.id.replace(/^field-/, '');
    var maskInfo = fieldMasks[fieldId];

    if (!maskInfo) {
      return;
    }

    var mask = maskInfo.pattern;
    var rawValue = extractRawInput(input.value);
    var formatted = applyMask(rawValue, mask);

    // Preserve cursor position intelligently
    var cursorBefore = input.selectionStart;
    var lengthBefore = input.value.length;

    input.value = formatted;

    // Calculate new cursor position
    var lengthAfter = formatted.length;
    var cursorAfter = cursorBefore + (lengthAfter - lengthBefore);

    // Ensure cursor is after any literal characters that were auto-inserted
    if (cursorAfter < formatted.length) {
      while (cursorAfter < mask.length && !isPlaceholder(mask[cursorAfter])) {
        cursorAfter++;
      }
    }

    // Clamp cursor position
    cursorAfter = Math.max(0, Math.min(cursorAfter, formatted.length));

    input.setSelectionRange(cursorAfter, cursorAfter);

    // Update visual state
    updateFieldState(input, maskInfo);
  }

  /**
   * Handle paste events by stripping non-essential characters and re-applying the mask.
   * @param {ClipboardEvent} event
   */
  function handlePaste(event) {
    event.preventDefault();

    var input = event.target;
    var fieldId = input.id.replace(/^field-/, '');
    var maskInfo = fieldMasks[fieldId];

    if (!maskInfo) {
      return;
    }

    var mask = maskInfo.pattern;
    var pastedText = '';

    if (event.clipboardData && event.clipboardData.getData) {
      pastedText = event.clipboardData.getData('text/plain');
    } else if (window.clipboardData) {
      // Legacy IE support
      pastedText = window.clipboardData.getData('Text');
    }

    if (!pastedText) {
      return;
    }

    // Get current value parts around selection
    var selStart = input.selectionStart;
    var selEnd = input.selectionEnd;
    var currentRaw = extractRawInput(input.value);

    // Figure out how many raw chars are before the selection start
    var rawBefore = '';
    var rawCharCount = 0;
    for (var i = 0; i < selStart && i < mask.length; i++) {
      if (isPlaceholder(mask[i]) && i < input.value.length) {
        rawBefore += input.value[i];
        rawCharCount++;
      }
    }

    // Figure out raw chars after the selection end
    var rawAfter = '';
    for (var j = selEnd; j < input.value.length && j < mask.length; j++) {
      if (isPlaceholder(mask[j])) {
        rawAfter += input.value[j];
      }
    }

    // Combine: before + pasted (stripped) + after
    var pastedRaw = extractRawInput(pastedText);
    var combinedRaw = rawBefore + pastedRaw + rawAfter;

    // Apply mask to combined value
    var formatted = applyMask(combinedRaw, mask);
    input.value = formatted;

    // Position cursor after the pasted content
    var newRawLength = rawBefore.length + pastedRaw.length;
    var newCursorPos = 0;
    var placeholdersSeen = 0;
    for (var k = 0; k < mask.length; k++) {
      if (isPlaceholder(mask[k])) {
        placeholdersSeen++;
        if (placeholdersSeen > newRawLength) {
          break;
        }
      }
      newCursorPos = k + 1;
    }

    newCursorPos = Math.min(newCursorPos, formatted.length);
    input.setSelectionRange(newCursorPos, newCursorPos);

    // Update visual state
    updateFieldState(input, maskInfo);

    // Dispatch change event so frameworks pick up the value change
    dispatchChangeEvent(input);
  }

  /**
   * Update the visual state of a masked field (CSS classes for completeness).
   * @param {HTMLInputElement} input
   * @param {object} maskInfo
   */
  function updateFieldState(input, maskInfo) {
    var mask = maskInfo.pattern;
    var value = input.value;

    input.classList.remove('qf-mask-incomplete', 'qf-mask-error');

    if (value.length === 0) {
      // Empty field - no special state
      return;
    }

    if (!isMaskComplete(value, mask)) {
      input.classList.add('qf-mask-incomplete');
    }
  }

  /**
   * Dispatch a synthetic 'change' event on an element.
   * @param {HTMLElement} element
   */
  function dispatchChangeEvent(element) {
    var event;
    if (typeof Event === 'function') {
      event = new Event('change', { bubbles: true });
    } else {
      // IE fallback
      event = document.createEvent('Event');
      event.initEvent('change', true, true);
    }
    element.dispatchEvent(event);
  }

  // ==================== Field Setup & Teardown ====================

  /**
   * Attach mask listeners to a single input field.
   * @param {string} fieldId - The QuikForms field ID
   * @param {string} maskPattern - The resolved mask pattern
   * @param {object} [options] - Additional options (label, required, etc.)
   */
  function attachMask(fieldId, maskPattern, options) {
    var input = document.getElementById('field-' + fieldId);
    if (!input) {
      console.warn('QuikForms InputMasking: Field element not found for id "' + fieldId + '"');
      return;
    }

    // Skip non-text input types
    var inputType = (input.type || '').toLowerCase();
    var validTypes = ['text', 'tel', 'search', ''];
    if (input.tagName !== 'INPUT' || validTypes.indexOf(inputType) === -1) {
      console.warn('QuikForms InputMasking: Field "' + fieldId + '" is not a text input. Skipping mask.');
      return;
    }

    var resolvedMask = resolveMask(maskPattern);
    if (!resolvedMask) {
      return;
    }

    // Store mask info
    var maskInfo = {
      fieldId: fieldId,
      pattern: resolvedMask,
      label: (options && options.label) || fieldId,
      required: (options && options.required) || false,
      originalPlaceholder: input.placeholder || ''
    };
    fieldMasks[fieldId] = maskInfo;

    // Set placeholder to show expected format
    input.placeholder = buildPlaceholder(resolvedMask);

    // Add CSS class for consistent styling
    input.classList.add('qf-masked-field');

    // Set maxlength to mask length to prevent over-typing
    input.setAttribute('maxlength', String(resolvedMask.length));

    // Store data attribute for identification
    input.setAttribute('data-qf-masked', 'true');
    input.setAttribute('data-qf-mask-pattern', resolvedMask);

    // Remove existing listeners if re-attaching
    detachMask(fieldId);

    // Create bound handlers
    var handlers = {
      keydown: handleKeyDown,
      input: handleInput,
      paste: handlePaste
    };

    input.addEventListener('keydown', handlers.keydown);
    input.addEventListener('input', handlers.input);
    input.addEventListener('paste', handlers.paste);

    // Store handlers for cleanup
    fieldListeners[fieldId] = {
      element: input,
      handlers: handlers
    };

    // If the field already has a value, apply the mask to it
    if (input.value) {
      var rawValue = extractRawInput(input.value);
      input.value = applyMask(rawValue, resolvedMask);
      updateFieldState(input, maskInfo);
    }
  }

  /**
   * Remove mask listeners from a field.
   * @param {string} fieldId
   */
  function detachMask(fieldId) {
    var listener = fieldListeners[fieldId];
    if (!listener) {
      return;
    }

    var el = listener.element;
    var handlers = listener.handlers;

    if (el) {
      el.removeEventListener('keydown', handlers.keydown);
      el.removeEventListener('input', handlers.input);
      el.removeEventListener('paste', handlers.paste);
    }

    delete fieldListeners[fieldId];
  }

  // ==================== Configuration Reading ====================

  /**
   * Read mask configuration from multiple sources:
   * 1. Global window.QuikFormsInputMaskingConfig
   * 2. Plugin configuration passed via formData
   * 3. data-mask attributes on field elements
   * @param {object} context - Hook context with formData, fieldValues, locale
   * @returns {object} Map of fieldId -> { mask, label, required }
   */
  function readConfiguration(context) {
    var config = {};

    // Source 1: Global configuration object
    if (window.QuikFormsInputMaskingConfig && window.QuikFormsInputMaskingConfig.fields) {
      var globalFields = window.QuikFormsInputMaskingConfig.fields;
      for (var fieldId in globalFields) {
        if (globalFields.hasOwnProperty(fieldId)) {
          var entry = globalFields[fieldId];
          if (typeof entry === 'string') {
            config[fieldId] = { mask: entry };
          } else if (entry && entry.mask) {
            config[fieldId] = {
              mask: entry.mask,
              label: entry.label || null,
              required: entry.required || false
            };
          }
        }
      }
    }

    // Source 2: Plugin configuration from formData (Configuration_Schema__c)
    if (context && context.formData && context.formData.pluginConfig) {
      var pConfig = context.formData.pluginConfig.inputMasking;
      if (pConfig && pConfig.fields) {
        var pFields = pConfig.fields;
        for (var pFieldId in pFields) {
          if (pFields.hasOwnProperty(pFieldId)) {
            var pEntry = pFields[pFieldId];
            if (typeof pEntry === 'string') {
              config[pFieldId] = config[pFieldId] || {};
              config[pFieldId].mask = pEntry;
            } else if (pEntry && pEntry.mask) {
              config[pFieldId] = config[pFieldId] || {};
              config[pFieldId].mask = pEntry.mask;
              if (pEntry.label) { config[pFieldId].label = pEntry.label; }
              if (pEntry.required !== undefined) { config[pFieldId].required = pEntry.required; }
            }
          }
        }
      }
    }

    // Source 3: data-mask attributes on field elements in the DOM
    var maskedElements = document.querySelectorAll('[data-mask]');
    for (var i = 0; i < maskedElements.length; i++) {
      var el = maskedElements[i];
      var elId = (el.id || '').replace(/^field-/, '');
      if (elId && el.getAttribute('data-mask')) {
        config[elId] = config[elId] || {};
        config[elId].mask = el.getAttribute('data-mask');
        if (el.getAttribute('data-mask-label')) {
          config[elId].label = el.getAttribute('data-mask-label');
        }
      }
    }

    // Also scan formData.fields for any with data attributes or field metadata
    if (context && context.formData && context.formData.fields) {
      var fields = context.formData.fields;
      for (var f = 0; f < fields.length; f++) {
        var field = fields[f];
        if (field.maskPattern) {
          config[field.id] = config[field.id] || {};
          config[field.id].mask = field.maskPattern;
          config[field.id].label = config[field.id].label || field.label || field.id;
          config[field.id].required = config[field.id].required || field.required || false;
        }
      }
    }

    return config;
  }

  /**
   * Enrich configuration with field label and required info from formData.
   * @param {object} config - Map of fieldId -> { mask, label, required }
   * @param {object} context - Hook context
   */
  function enrichConfigFromFormData(config, context) {
    if (!context || !context.formData || !context.formData.fields) {
      return;
    }

    var fields = context.formData.fields;
    for (var i = 0; i < fields.length; i++) {
      var field = fields[i];
      if (config[field.id]) {
        if (!config[field.id].label) {
          config[field.id].label = field.label || field.id;
        }
        if (config[field.id].required === undefined) {
          config[field.id].required = field.required || false;
        }
      }
    }
  }

  // ==================== Plugin Hook Implementations ====================

  /**
   * onFormLoad hook: Initialize masks on all configured fields.
   * Reads configuration, attaches keydown/input/paste listeners, sets placeholders.
   * @param {object} context - { formData, fieldValues, locale }
   * @returns {object|null}
   */
  function onFormLoad(context) {
    injectCSS();

    // Read and merge configuration from all sources
    var config = readConfiguration(context);
    enrichConfigFromFormData(config, context);

    pluginConfig = config;

    // Track how many fields were masked
    var maskedCount = 0;

    for (var fieldId in config) {
      if (config.hasOwnProperty(fieldId)) {
        var fieldConfig = config[fieldId];
        attachMask(fieldId, fieldConfig.mask, {
          label: fieldConfig.label,
          required: fieldConfig.required
        });
        maskedCount++;
      }
    }

    if (maskedCount > 0) {
      console.log('QuikForms InputMasking: Initialized masks on ' + maskedCount + ' field(s)');
    }

    return null;
  }

  /**
   * onFieldChange hook: Re-apply mask after programmatic value changes.
   * @param {object} context - { fieldId, field, value, formData, locale }
   * @returns {object|null}
   */
  function onFieldChange(context) {
    if (!context || !context.fieldId) {
      return null;
    }

    var maskInfo = fieldMasks[context.fieldId];
    if (!maskInfo) {
      return null;
    }

    var input = document.getElementById('field-' + context.fieldId);
    if (!input) {
      return null;
    }

    // Re-apply the mask to the current value
    var rawValue = extractRawInput(input.value);
    var formatted = applyMask(rawValue, maskInfo.pattern);

    if (input.value !== formatted) {
      input.value = formatted;
    }

    updateFieldState(input, maskInfo);

    return null;
  }

  /**
   * onFieldBlur hook: Finalize formatting when user leaves the field.
   * Ensures the value is properly formatted and updates visual state.
   * @param {object} context - { fieldId, field, value, formData, locale }
   * @returns {object|null}
   */
  function onFieldBlur(context) {
    if (!context || !context.fieldId) {
      return null;
    }

    var maskInfo = fieldMasks[context.fieldId];
    if (!maskInfo) {
      return null;
    }

    var input = document.getElementById('field-' + context.fieldId);
    if (!input) {
      return null;
    }

    var value = input.value;

    // If field is empty, remove any visual states
    if (!value || value.trim() === '') {
      input.classList.remove('qf-mask-incomplete', 'qf-mask-error');
      return null;
    }

    // Re-apply mask to ensure final formatting is correct
    var rawValue = extractRawInput(value);
    var formatted = applyMask(rawValue, maskInfo.pattern);

    if (input.value !== formatted) {
      input.value = formatted;
    }

    // Update visual state
    updateFieldState(input, maskInfo);

    return null;
  }

  /**
   * onValidate hook: Validate mask completeness for all masked fields.
   * Returns errors for incomplete masks and modifiedValues with stripped
   * (unformatted) values so Salesforce receives clean data.
   * @param {object} context - { formData, fieldValues, locale }
   * @returns {object} { errors: [], warnings: [], modifiedValues: {} }
   */
  function onValidate(context) {
    var errors = [];
    var warnings = [];
    var modifiedValues = {};

    for (var fieldId in fieldMasks) {
      if (!fieldMasks.hasOwnProperty(fieldId)) {
        continue;
      }

      var maskInfo = fieldMasks[fieldId];
      var input = document.getElementById('field-' + fieldId);

      if (!input) {
        continue;
      }

      var value = input.value;
      var mask = maskInfo.pattern;
      var label = maskInfo.label || fieldId;

      // Skip empty non-required fields
      if (!value || value.trim() === '') {
        if (maskInfo.required) {
          // Required validation is handled by the core framework, not this plugin
          continue;
        }
        continue;
      }

      // Check mask completeness
      if (!isMaskComplete(value, mask)) {
        var expectedLength = countPlaceholders(mask);
        var actualRaw = extractRawInput(value);
        var actualLength = actualRaw.length;

        errors.push({
          fieldId: fieldId,
          message: label + ' is incomplete. Expected format: ' + buildPlaceholder(mask) +
            ' (' + actualLength + ' of ' + expectedLength + ' characters entered)'
        });

        // Add error styling
        input.classList.add('qf-mask-error');
        input.classList.remove('qf-mask-incomplete');
      } else {
        // Mask is complete - strip formatting for Salesforce
        var strippedValue = stripMask(value, mask);
        modifiedValues[fieldId] = strippedValue;

        // Remove any error styling
        input.classList.remove('qf-mask-error', 'qf-mask-incomplete');
      }
    }

    var result = {};

    if (errors.length > 0) {
      result.errors = errors;
    }

    if (warnings.length > 0) {
      result.warnings = warnings;
    }

    if (Object.keys(modifiedValues).length > 0) {
      result.modifiedValues = modifiedValues;
    }

    return result;
  }

  // ==================== Plugin Registration ====================

  // Register with the QuikForms plugin framework if available
  if (typeof QuikFormsPlugins !== 'undefined' && QuikFormsPlugins.register) {
    QuikFormsPlugins.register('inputMasking', {
      name: 'Input Masking',
      version: '1.0.0',
      description: 'Applies format masks to text fields as users type. Supports phone numbers, SSNs, EINs, credit cards, postal codes, dates, and custom patterns.',
      hooks: {
        onFormLoad: onFormLoad,
        onFieldChange: onFieldChange,
        onFieldBlur: onFieldBlur,
        onValidate: onValidate
      }
    });
  }

  // ==================== Public API ====================

  // Expose utility functions for external use and testing
  window.QuikFormsInputMasking = {
    // Core functions
    applyMask: applyMask,
    stripMask: stripMask,
    extractRawInput: extractRawInput,
    isMaskComplete: isMaskComplete,
    buildPlaceholder: buildPlaceholder,
    resolveMask: resolveMask,

    // Predefined masks
    predefinedMasks: PREDEFINED_MASKS,

    // Field management
    attachMask: attachMask,
    detachMask: detachMask,

    // State access (read-only references)
    getFieldMasks: function() { return fieldMasks; },
    getFieldMask: function(fieldId) { return fieldMasks[fieldId] || null; },

    // Hook functions (for direct invocation or testing)
    hooks: {
      onFormLoad: onFormLoad,
      onFieldChange: onFieldChange,
      onFieldBlur: onFieldBlur,
      onValidate: onValidate
    }
  };

})();

/**
 * Entry point function called by the QuikForms plugin framework
 * after the static resource is loaded. The IIFE above handles
 * self-registration so this function is a no-op safety net.
 * @param {object} registry - The QuikFormsPlugins registry instance
 */
function initQuikFormsInputMasking(registry) {
  // Plugin self-registers in the IIFE above.
  // This entry point exists as a safety net for the plugin loader.
  // If the plugin did not register (e.g., QuikFormsPlugins was not
  // available at IIFE execution time), register now.
  if (registry && registry.register && !registry.plugins['inputMasking']) {
    var api = window.QuikFormsInputMasking;
    if (api && api.hooks) {
      registry.register('inputMasking', {
        name: 'Input Masking',
        version: '1.0.0',
        description: 'Applies format masks to text fields as users type.',
        hooks: {
          onFormLoad: api.hooks.onFormLoad,
          onFieldChange: api.hooks.onFieldChange,
          onFieldBlur: api.hooks.onFieldBlur,
          onValidate: api.hooks.onValidate
        }
      });
    }
  }
}
