/**
 * @description Test class for QuikFormsExternalVerificationPlugin.
 * Provides comprehensive test coverage for plugin initialization,
 * metadata, and callout execution scenarios including success,
 * server error, timeout, Named Credential usage, and edge cases.
 *
 * @author QuikForms
 * @since 1.0
 */
@isTest
private class QuikFormsExternalVerificationPluginTest {

    // ======================== Mock HTTP Response ========================

    /**
     * @description Configurable HTTP callout mock that returns a preset
     * status code and body. Used to simulate external API responses.
     */
    private class MockHttpResponse implements HttpCalloutMock {
        private Integer statusCode;
        private String body;

        MockHttpResponse(Integer statusCode, String body) {
            this.statusCode = statusCode;
            this.body = body;
        }

        public HTTPResponse respond(HTTPRequest req) {
            HttpResponse resp = new HttpResponse();
            resp.setStatusCode(this.statusCode);
            resp.setBody(this.body);
            resp.setHeader('Content-Type', 'application/json');
            return resp;
        }
    }

    /**
     * @description HTTP callout mock that throws a CalloutException
     * to simulate network timeouts or connection failures.
     */
    private class MockCalloutException implements HttpCalloutMock {
        public HTTPResponse respond(HTTPRequest req) {
            throw new System.CalloutException('Read timed out');
        }
    }

    /**
     * @description HTTP callout mock that captures the request for
     * inspection in test assertions.
     */
    private class MockRequestCapture implements HttpCalloutMock {
        public String capturedEndpoint;
        public String capturedMethod;
        public String capturedBody;
        public Integer capturedTimeout;

        public HTTPResponse respond(HTTPRequest req) {
            this.capturedEndpoint = req.getEndpoint();
            this.capturedMethod = req.getMethod();
            this.capturedBody = req.getBody();
            this.capturedTimeout = req.getTimeout();

            HttpResponse resp = new HttpResponse();
            resp.setStatusCode(200);
            resp.setBody('{"verified":true}');
            return resp;
        }
    }

    // ======================== Helpers ========================

    /**
     * @description Creates a standard plugin configuration map with
     * an endpoint and verification type paths.
     */
    private static Map<String, Object> createStandardConfig() {
        Map<String, Object> verificationTypes = new Map<String, Object>{
            'email' => '/verify/email',
            'phone' => '/verify/phone',
            'domain' => '/verify/domain'
        };

        return new Map<String, Object>{
            'endpoint' => 'https://api.example.com',
            'timeoutMs' => 5000,
            'verificationTypes' => verificationTypes
        };
    }

    /**
     * @description Creates a QuikFormsCalloutRequest with the given
     * params serialized as the request body.
     */
    private static QuikFormsCalloutRequest createCalloutRequest(Map<String, Object> params) {
        QuikFormsCalloutRequest request = new QuikFormsCalloutRequest();
        request.endpoint = 'https://api.example.com';
        request.method = 'POST';
        request.body = JSON.serialize(params);
        request.timeoutMs = 5000;
        return request;
    }

    // ======================== Tests: getPluginInfo ========================

    @isTest
    static void testGetPluginInfo() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();

        Test.startTest();
        QuikFormsPluginInfo info = plugin.getPluginInfo();
        Test.stopTest();

        System.assertEquals('External Verification', info.name, 'Plugin name should be External Verification');
        System.assertEquals('1.0.0', info.version, 'Plugin version should be 1.0.0');
        System.assertEquals(true, info.supportsCallouts, 'Plugin should support callouts');
        System.assertNotEquals(null, info.description, 'Plugin should have a description');
        System.assert(info.description.length() > 0, 'Description should not be empty');
        System.assert(info.supportedHooks.contains('onFieldBlur'), 'Should list onFieldBlur as supported hook');
    }

    // ======================== Tests: initialize and isReady ========================

    @isTest
    static void testInitializeAndIsReady() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();

        // Before initialization
        System.assertEquals(false, plugin.isReady(), 'Plugin should not be ready before initialization');

        Test.startTest();
        plugin.initialize(createStandardConfig());
        Test.stopTest();

        System.assertEquals(true, plugin.isReady(), 'Plugin should be ready after initialization');
    }

    @isTest
    static void testInitializeWithNullConfig() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();

        Test.startTest();
        plugin.initialize(null);
        Test.stopTest();

        System.assertEquals(true, plugin.isReady(), 'Plugin should be ready even with null config');
    }

    @isTest
    static void testInitializeWithEmptyConfig() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();

        Test.startTest();
        plugin.initialize(new Map<String, Object>());
        Test.stopTest();

        System.assertEquals(true, plugin.isReady(), 'Plugin should be ready with empty config');
    }

    // ======================== Tests: executeCallout - Success ========================

    @isTest
    static void testExecuteCalloutSuccess() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        String responseBody = '{"verified":true,"message":"Email is valid"}';
        Test.setMock(HttpCalloutMock.class, new MockHttpResponse(200, responseBody));

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'test@example.com',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(true, response.success, 'Response should be successful');
        System.assertEquals(200, response.statusCode, 'Status code should be 200');
        System.assertEquals(responseBody, response.body, 'Response body should match mock');
    }

    @isTest
    static void testExecuteCalloutSuccessWithDifferentVerificationType() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        String responseBody = '{"verified":true,"message":"Phone is valid"}';
        Test.setMock(HttpCalloutMock.class, new MockHttpResponse(200, responseBody));

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'phone_field',
            'value' => '+15551234567',
            'verificationType' => 'phone'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(true, response.success, 'Response should be successful for phone verification');
        System.assertEquals(200, response.statusCode, 'Status code should be 200');
    }

    // ======================== Tests: executeCallout - Server Error ========================

    @isTest
    static void testExecuteCalloutServerError() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        Test.setMock(HttpCalloutMock.class, new MockHttpResponse(500, '{"error":"Internal Server Error"}'));

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'test@example.com',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(false, response.success, 'Response should indicate failure');
        System.assertNotEquals(null, response.errorMessage, 'Should have an error message');
        System.assert(
            response.errorMessage.contains('500'),
            'Error message should include the status code: ' + response.errorMessage
        );
    }

    @isTest
    static void testExecuteCalloutClientError() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        Test.setMock(HttpCalloutMock.class, new MockHttpResponse(400, '{"error":"Bad Request"}'));

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'invalid-email',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(false, response.success, 'Response should indicate failure for 400');
        System.assert(
            response.errorMessage.contains('400'),
            'Error message should include status code 400: ' + response.errorMessage
        );
    }

    // ======================== Tests: executeCallout - Timeout ========================

    @isTest
    static void testExecuteCalloutTimeout() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        Test.setMock(HttpCalloutMock.class, new MockCalloutException());

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'test@example.com',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(false, response.success, 'Response should indicate failure on timeout');
        System.assertNotEquals(null, response.errorMessage, 'Should have an error message');
        System.assert(
            response.errorMessage.contains('callout failed') || response.errorMessage.contains('timed out'),
            'Error message should indicate callout failure: ' + response.errorMessage
        );
    }

    // ======================== Tests: executeCallout - Named Credential ========================

    @isTest
    static void testExecuteCalloutWithNamedCredential() {
        Map<String, Object> config = createStandardConfig();
        config.put('namedCredential', 'Verification_API');

        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(config);

        MockRequestCapture captureMock = new MockRequestCapture();
        Test.setMock(HttpCalloutMock.class, captureMock);

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'test@example.com',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(true, response.success, 'Response should be successful');
        System.assert(
            captureMock.capturedEndpoint.startsWith('callout:Verification_API'),
            'Endpoint should use Named Credential prefix: ' + captureMock.capturedEndpoint
        );
        System.assert(
            captureMock.capturedEndpoint.contains('/verify/email'),
            'Endpoint should include verification type path: ' + captureMock.capturedEndpoint
        );
    }

    // ======================== Tests: executeCallout - Empty/Null Body ========================

    @isTest
    static void testExecuteCalloutEmptyBody() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        // No mock needed -- we expect failure before the HTTP call
        QuikFormsCalloutRequest request = new QuikFormsCalloutRequest();
        request.body = null;

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(false, response.success, 'Response should indicate failure for empty body');
        System.assertNotEquals(null, response.errorMessage, 'Should have an error message');
        System.assert(
            response.errorMessage.contains('required'),
            'Error should mention that the value is required: ' + response.errorMessage
        );
    }

    @isTest
    static void testExecuteCalloutBlankValue() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => '',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(false, response.success, 'Response should indicate failure for blank value');
        System.assert(
            response.errorMessage.contains('required'),
            'Error should mention that the value is required: ' + response.errorMessage
        );
    }

    @isTest
    static void testExecuteCalloutEmptyStringBody() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        QuikFormsCalloutRequest request = new QuikFormsCalloutRequest();
        request.body = '';

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(false, response.success, 'Response should indicate failure for empty string body');
        System.assertNotEquals(null, response.errorMessage, 'Should have an error message');
    }

    @isTest
    static void testExecuteCalloutInvalidJsonBody() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        QuikFormsCalloutRequest request = new QuikFormsCalloutRequest();
        request.body = 'not valid json{{{';

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(false, response.success, 'Response should indicate failure for invalid JSON body');
    }

    // ======================== Tests: executeCallout - Null Request ========================

    @isTest
    static void testExecuteCalloutNullRequest() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(null);
        Test.stopTest();

        System.assertEquals(false, response.success, 'Response should indicate failure for null request');
        System.assertNotEquals(null, response.errorMessage, 'Should have an error message');
    }

    // ======================== Tests: executeCallout - Verification Type Fallback ========================

    @isTest
    static void testExecuteCalloutUnknownVerificationType() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        String responseBody = '{"verified":false,"message":"Unknown type"}';
        Test.setMock(HttpCalloutMock.class, new MockHttpResponse(200, responseBody));

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'custom_field',
            'value' => 'some value',
            'verificationType' => 'unknownType'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        // Should still succeed (HTTP 200) even if verification type is not in the configured paths
        System.assertEquals(true, response.success, 'Response should be successful even for unknown verification type');
    }

    @isTest
    static void testExecuteCalloutMissingVerificationType() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        String responseBody = '{"verified":true}';
        Test.setMock(HttpCalloutMock.class, new MockHttpResponse(200, responseBody));

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'custom_field',
            'value' => 'test value'
            // No verificationType
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(true, response.success, 'Response should be successful with default verification type');
    }

    // ======================== Tests: executeCallout - Config Edge Cases ========================

    @isTest
    static void testExecuteCalloutNoVerificationTypesInConfig() {
        Map<String, Object> config = new Map<String, Object>{
            'endpoint' => 'https://api.example.com'
        };

        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(config);

        String responseBody = '{"verified":true}';
        Test.setMock(HttpCalloutMock.class, new MockHttpResponse(200, responseBody));

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'test@example.com',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(true, response.success, 'Should succeed without verificationTypes in config');
    }

    @isTest
    static void testExecuteCalloutEndpointFromRequest() {
        // Config with no endpoint, should fall back to request.endpoint
        Map<String, Object> config = new Map<String, Object>();

        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(config);

        MockRequestCapture captureMock = new MockRequestCapture();
        Test.setMock(HttpCalloutMock.class, captureMock);

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'test@example.com',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = new QuikFormsCalloutRequest();
        request.endpoint = 'https://fallback.example.com';
        request.body = JSON.serialize(params);
        request.timeoutMs = 3000;

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(true, response.success, 'Response should be successful');
        System.assert(
            captureMock.capturedEndpoint.startsWith('https://fallback.example.com'),
            'Should use request endpoint as fallback: ' + captureMock.capturedEndpoint
        );
    }

    @isTest
    static void testExecuteCalloutConfigTimeout() {
        Map<String, Object> config = new Map<String, Object>{
            'endpoint' => 'https://api.example.com',
            'timeoutMs' => 8000
        };

        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(config);

        MockRequestCapture captureMock = new MockRequestCapture();
        Test.setMock(HttpCalloutMock.class, captureMock);

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'test@example.com',
            'verificationType' => 'email'
        };
        // Request with null timeoutMs to force config fallback
        QuikFormsCalloutRequest request = new QuikFormsCalloutRequest();
        request.endpoint = 'https://api.example.com';
        request.body = JSON.serialize(params);
        request.timeoutMs = null;

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(true, response.success, 'Response should be successful');
        System.assertEquals(8000, captureMock.capturedTimeout, 'Should use config timeout when request timeout is null');
    }

    // ======================== Tests: HTTP 201/204 Success ========================

    @isTest
    static void testExecuteCallout201Created() {
        QuikFormsExternalVerificationPlugin plugin = new QuikFormsExternalVerificationPlugin();
        plugin.initialize(createStandardConfig());

        Test.setMock(HttpCalloutMock.class, new MockHttpResponse(201, '{"verified":true}'));

        Map<String, Object> params = new Map<String, Object>{
            'fieldId' => 'email_field',
            'value' => 'test@example.com',
            'verificationType' => 'email'
        };
        QuikFormsCalloutRequest request = createCalloutRequest(params);

        Test.startTest();
        QuikFormsCalloutResponse response = plugin.executeCallout(request);
        Test.stopTest();

        System.assertEquals(true, response.success, '201 should be treated as success');
        System.assertEquals(201, response.statusCode, 'Status code should be 201');
    }
}
