/**
 * @description External Verification plugin for QuikForms.
 * Verifies form field values against external APIs via HTTP callouts.
 * Implements both IQuikFormsCalloutHandler (for HTTP callouts) and
 * IQuikFormsPlugin (base plugin interface).
 *
 * Configuration keys (passed via initialize()):
 *   endpoint           - Base URL for the verification API
 *   namedCredential    - Optional Named Credential for authenticated callouts
 *   timeoutMs          - Callout timeout in milliseconds (default: 5000)
 *   verificationTypes  - Map of verification type to endpoint path
 *                        e.g., { "email": "/verify/email", "domain": "/verify/domain", "phone": "/verify/phone" }
 *
 * @author QuikForms
 * @since 1.0
 */
global class QuikFormsExternalVerificationPlugin implements IQuikFormsCalloutHandler, IQuikFormsPlugin {

    private static final Integer DEFAULT_TIMEOUT_MS = 5000;

    private Map<String, Object> config;
    private Boolean initialized = false;

    /**
     * @description Returns metadata about this plugin.
     * @return QuikFormsPluginInfo containing plugin name, version, description, and capabilities
     */
    global QuikFormsPluginInfo getPluginInfo() {
        QuikFormsPluginInfo info = new QuikFormsPluginInfo();
        info.name = 'External Verification';
        info.version = '1.0.0';
        info.description = 'Verifies form data against external APIs on field blur';
        info.supportsCallouts = true;
        info.supportedHooks = new List<String>{ 'onFieldBlur' };
        return info;
    }

    /**
     * @description Called once when the plugin is first loaded.
     * Stores the configuration map for use during callout execution.
     * @param configuration Map of configuration options from the plugin metadata
     */
    global void initialize(Map<String, Object> configuration) {
        this.config = configuration != null ? configuration : new Map<String, Object>();
        this.initialized = true;
    }

    /**
     * @description Returns true if the plugin has been initialized and is ready to execute.
     * @return Boolean indicating readiness
     */
    global Boolean isReady() {
        return this.initialized;
    }

    /**
     * @description Execute a verification callout to an external API.
     * Parses the request body for fieldId, value, and verificationType,
     * builds the appropriate endpoint URL, makes the HTTP callout,
     * and returns the result.
     *
     * @param request QuikFormsCalloutRequest containing the verification parameters in its body
     * @return QuikFormsCalloutResponse with the verification result
     */
    global QuikFormsCalloutResponse executeCallout(QuikFormsCalloutRequest request) {
        try {
            // Parse request parameters
            Map<String, Object> params = parseRequestBody(request);
            String value = (String) params.get('value');
            String verificationType = (String) params.get('verificationType');

            // Validate required parameters
            if (String.isBlank(value)) {
                return QuikFormsCalloutResponse.failure('Verification value is required');
            }
            if (String.isBlank(verificationType)) {
                verificationType = 'default';
            }

            // Build and execute the HTTP request
            HttpRequest httpReq = buildHttpRequest(request, value, verificationType);
            Http http = new Http();
            HttpResponse httpResp = http.send(httpReq);

            // Process response
            Integer statusCode = httpResp.getStatusCode();
            if (statusCode >= 200 && statusCode < 300) {
                return QuikFormsCalloutResponse.success(statusCode, httpResp.getBody());
            } else {
                return QuikFormsCalloutResponse.failure(
                    'Verification service returned status ' + statusCode
                );
            }
        } catch (System.CalloutException e) {
            return QuikFormsCalloutResponse.failure(
                'Verification callout failed: ' + e.getMessage()
            );
        } catch (Exception e) {
            return QuikFormsCalloutResponse.failure(
                'Verification failed: ' + e.getMessage()
            );
        }
    }

    /* ------------------------------------------------------------------ */
    /*  Private helpers                                                    */
    /* ------------------------------------------------------------------ */

    /**
     * @description Parse the request body JSON into a Map.
     * Returns an empty map if the body is null or invalid JSON.
     * @param request The callout request
     * @return Map of parsed parameters
     */
    private Map<String, Object> parseRequestBody(QuikFormsCalloutRequest request) {
        if (request == null || String.isBlank(request.body)) {
            return new Map<String, Object>();
        }
        try {
            return (Map<String, Object>) JSON.deserializeUntyped(request.body);
        } catch (Exception e) {
            return new Map<String, Object>();
        }
    }

    /**
     * @description Build the HttpRequest for the verification callout.
     * Resolves the endpoint from configuration (Named Credential or direct URL),
     * sets method, headers, timeout, and body.
     *
     * @param request The original callout request
     * @param value The field value to verify
     * @param verificationType The type of verification to perform
     * @return Configured HttpRequest ready to send
     */
    private HttpRequest buildHttpRequest(QuikFormsCalloutRequest request, String value, String verificationType) {
        HttpRequest httpReq = new HttpRequest();

        // Resolve endpoint path for this verification type
        String path = resolveVerificationPath(verificationType);

        // Determine base endpoint and apply Named Credential if configured
        String namedCred = getConfigString('namedCredential');
        if (String.isNotBlank(namedCred)) {
            httpReq.setEndpoint('callout:' + namedCred + path);
        } else {
            String baseEndpoint = getConfigString('endpoint');
            if (String.isBlank(baseEndpoint) && request != null) {
                baseEndpoint = request.endpoint;
            }
            if (String.isBlank(baseEndpoint)) {
                baseEndpoint = '';
            }
            httpReq.setEndpoint(baseEndpoint + path);
        }

        httpReq.setMethod('POST');
        httpReq.setHeader('Content-Type', 'application/json');

        // Resolve timeout: request-level > config-level > default
        Integer timeoutMs = DEFAULT_TIMEOUT_MS;
        if (request != null && request.timeoutMs != null && request.timeoutMs > 0) {
            timeoutMs = request.timeoutMs;
        } else {
            Integer configTimeout = getConfigInteger('timeoutMs');
            if (configTimeout != null && configTimeout > 0) {
                timeoutMs = configTimeout;
            }
        }
        httpReq.setTimeout(timeoutMs);

        // Build request body
        Map<String, String> bodyMap = new Map<String, String>{
            'value' => value,
            'type' => verificationType
        };
        httpReq.setBody(JSON.serialize(bodyMap));

        return httpReq;
    }

    /**
     * @description Resolve the endpoint path for a given verification type
     * from the verificationTypes configuration map.
     * @param verificationType The verification type key
     * @return The endpoint path, or empty string if not configured
     */
    private String resolveVerificationPath(String verificationType) {
        if (!this.config.containsKey('verificationTypes')) {
            return '';
        }

        Object typesObj = this.config.get('verificationTypes');
        if (!(typesObj instanceof Map<String, Object>)) {
            return '';
        }

        Map<String, Object> types = (Map<String, Object>) typesObj;
        if (types.containsKey(verificationType)) {
            return String.valueOf(types.get(verificationType));
        }

        return '';
    }

    /**
     * @description Safely get a String value from the config map.
     * @param key The configuration key
     * @return The string value, or null if not found
     */
    private String getConfigString(String key) {
        if (this.config == null || !this.config.containsKey(key)) {
            return null;
        }
        Object val = this.config.get(key);
        return val != null ? String.valueOf(val) : null;
    }

    /**
     * @description Safely get an Integer value from the config map.
     * @param key The configuration key
     * @return The integer value, or null if not found or not a number
     */
    private Integer getConfigInteger(String key) {
        if (this.config == null || !this.config.containsKey(key)) {
            return null;
        }
        Object val = this.config.get(key);
        if (val instanceof Integer) {
            return (Integer) val;
        }
        if (val instanceof Decimal) {
            return ((Decimal) val).intValue();
        }
        try {
            return Integer.valueOf(String.valueOf(val));
        } catch (Exception e) {
            return null;
        }
    }
}
