/**
 * QuikForms Confirmation Step Plugin v1.0.0
 *
 * Intercepts form submission to display a read-only review modal so users
 * can verify their answers before the data is actually submitted.
 *
 * Two-phase submit flow:
 *   1. User clicks "Review & Submit" -> validation hook shows the modal and aborts.
 *   2. User clicks "Confirm & Submit" in the modal -> validation hook allows submission.
 *
 * Hooks used: onFormLoad, onValidate
 *
 * @license MIT
 */
(function () {
  'use strict';

  // ---------------------------------------------------------------------------
  // Internal state
  // ---------------------------------------------------------------------------
  var reviewConfirmed = false;
  var modalOverlay = null;

  // ---------------------------------------------------------------------------
  // Utility helpers
  // ---------------------------------------------------------------------------

  /**
   * Escape HTML entities to prevent XSS when rendering user-supplied values.
   */
  function escapeHtml(str) {
    if (str === null || str === undefined) return '';
    return String(str)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');
  }

  /**
   * Determine whether a field row is currently hidden (dependency-driven visibility).
   */
  function isFieldHidden(fieldId) {
    var row = document.getElementById('field-row-' + fieldId);
    if (!row) return true;
    return row.classList.contains('hidden') || row.style.display === 'none';
  }

  /**
   * Format a field value for display in the review summary.
   */
  function formatDisplayValue(field, rawValue) {
    // Checkbox (boolean) fields
    if (field.type === 'checkBox') {
      if (rawValue === 'true' || rawValue === true || rawValue === '1') return 'Yes';
      return 'No';
    }

    // Masked / password fields - show asterisks
    if (field.isMasked) {
      if (!rawValue) return '';
      return new Array(rawValue.length + 1).join('*');
    }

    // Multi-select checkboxes -> comma-separated (already stored that way, just tidy up)
    if (field.customStructureType === 'selectCheckboxes' && rawValue) {
      return String(rawValue).split(',').map(function (v) { return v.trim(); }).filter(Boolean).join(', ');
    }

    // File / attachment fields - show filename when available
    if (field.objectField === '$ATTACHMENT' || field.objectField === '$CONTENTDOCUMENT') {
      var fileInput = document.querySelector('#field-row-' + field.id + ' input[type="file"]');
      if (fileInput && fileInput.files && fileInput.files.length > 0) {
        var names = [];
        for (var i = 0; i < fileInput.files.length; i++) {
          names.push(fileInput.files[i].name);
        }
        return names.join(', ');
      }
      return rawValue || '';
    }

    // Dates and everything else - return as-is
    return rawValue != null ? String(rawValue) : '';
  }

  /**
   * Determine if a field should be shown in the review summary.
   */
  function shouldIncludeField(field, value) {
    // Skip custom HTML (non-data)
    if (field.type === 'customHTML') return false;

    // Skip dependency-hidden fields
    if (isFieldHidden(field.id)) return false;

    // Skip empty optional fields (but keep required empties so user notices)
    if (!field.isRequired && (value === '' || value === null || value === undefined)) return false;

    return true;
  }

  // ---------------------------------------------------------------------------
  // Modal rendering
  // ---------------------------------------------------------------------------

  /**
   * Build the summary table rows from field definitions and collected values.
   */
  function buildSummaryRows(fields, fieldValues) {
    var rows = '';

    for (var i = 0; i < fields.length; i++) {
      var field = fields[i];
      var rawValue = fieldValues[field.id];
      var displayValue = formatDisplayValue(field, rawValue);

      if (!shouldIncludeField(field, displayValue)) continue;

      var label = escapeHtml(field.label || field.id);
      var value = escapeHtml(displayValue) || '<span class="qf-cs-empty">--</span>';

      rows +=
        '<tr class="qf-cs-row">' +
          '<td class="qf-cs-label">' + label + '</td>' +
          '<td class="qf-cs-value">' + value + '</td>' +
        '</tr>';
    }

    return rows;
  }

  /**
   * Inject the modal CSS (once) into the document head.
   */
  var styleInjected = false;
  function injectStyles() {
    if (styleInjected) return;
    styleInjected = true;

    var css =
      /* Overlay */
      '.qf-cs-overlay {' +
        'position: fixed; inset: 0; z-index: 100000;' +
        'display: flex; align-items: center; justify-content: center;' +
        'background: rgba(0, 0, 0, 0.6);' +
        'backdrop-filter: blur(4px);' +
        'opacity: 0; transition: opacity 0.2s ease;' +
      '}' +
      '.qf-cs-overlay.qf-cs-visible { opacity: 1; }' +

      /* Modal card */
      '.qf-cs-modal {' +
        'position: relative;' +
        'width: 90%; max-width: 640px; max-height: 80vh;' +
        'display: flex; flex-direction: column;' +
        'background: var(--qf-bg-color, #0f172a);' +
        'color: var(--qf-text-color, #f1f5f9);' +
        'border: 1px solid var(--qf-input-border-color, #334155);' +
        'border-radius: var(--qf-radius-lg, 16px);' +
        'box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);' +
        'overflow: hidden;' +
        'transform: translateY(12px); transition: transform 0.2s ease;' +
      '}' +
      '.qf-cs-overlay.qf-cs-visible .qf-cs-modal { transform: translateY(0); }' +

      /* Header */
      '.qf-cs-header {' +
        'padding: 20px 24px;' +
        'font-size: 1.2em; font-weight: 700;' +
        'border-bottom: 1px solid var(--qf-input-border-color, #334155);' +
        'display: flex; align-items: center; justify-content: space-between;' +
      '}' +
      '.qf-cs-header-title { margin: 0; }' +
      '.qf-cs-close {' +
        'background: none; border: none; cursor: pointer;' +
        'color: var(--qf-text-color, #f1f5f9); font-size: 1.4em; line-height: 1;' +
        'padding: 4px 8px; border-radius: 4px; opacity: 0.7;' +
      '}' +
      '.qf-cs-close:hover, .qf-cs-close:focus { opacity: 1; outline: 2px solid var(--qf-highlight-color, #7c3aed); }' +

      /* Body (scrollable) */
      '.qf-cs-body {' +
        'padding: 16px 24px; overflow-y: auto; flex: 1 1 auto;' +
      '}' +

      /* Summary table */
      '.qf-cs-table {' +
        'width: 100%; border-collapse: collapse;' +
      '}' +
      '.qf-cs-row td {' +
        'padding: 10px 8px;' +
        'border-bottom: 1px solid var(--qf-input-border-color, rgba(51, 65, 85, 0.5));' +
        'vertical-align: top;' +
      '}' +
      '.qf-cs-label {' +
        'font-weight: 600; white-space: nowrap; width: 40%;' +
        'color: var(--qf-text-color, #f1f5f9); opacity: 0.85;' +
      '}' +
      '.qf-cs-value {' +
        'word-break: break-word;' +
      '}' +
      '.qf-cs-empty { opacity: 0.4; font-style: italic; }' +

      /* Footer */
      '.qf-cs-footer {' +
        'padding: 16px 24px;' +
        'display: flex; justify-content: flex-end; gap: 12px;' +
        'border-top: 1px solid var(--qf-input-border-color, #334155);' +
      '}' +

      /* Buttons */
      '.qf-cs-btn {' +
        'padding: 10px 20px; font-size: 0.95em; font-weight: 600;' +
        'border-radius: 8px; border: none; cursor: pointer;' +
        'transition: background 0.15s ease, box-shadow 0.15s ease;' +
      '}' +
      '.qf-cs-btn:focus-visible { outline: 2px solid var(--qf-highlight-color, #7c3aed); outline-offset: 2px; }' +
      '.qf-cs-btn-edit {' +
        'background: transparent;' +
        'color: var(--qf-text-color, #f1f5f9);' +
        'border: 1px solid var(--qf-input-border-color, #334155);' +
      '}' +
      '.qf-cs-btn-edit:hover { background: rgba(255,255,255,0.06); }' +
      '.qf-cs-btn-confirm {' +
        'background: var(--qf-button-color, #7c3aed);' +
        'color: #fff;' +
      '}' +
      '.qf-cs-btn-confirm:hover { box-shadow: 0 4px 16px rgba(124, 58, 237, 0.4); }' +

      /* Focus trap sentinel (visually hidden) */
      '.qf-cs-sr-only {' +
        'position: absolute; width: 1px; height: 1px; padding: 0;' +
        'margin: -1px; overflow: hidden; clip: rect(0,0,0,0);' +
        'white-space: nowrap; border: 0;' +
      '}' +

      /* Responsive */
      '@media (max-width: 480px) {' +
        '.qf-cs-modal { width: 96%; max-height: 90vh; }' +
        '.qf-cs-label { white-space: normal; width: 35%; }' +
        '.qf-cs-footer { flex-direction: column; }' +
        '.qf-cs-btn { width: 100%; text-align: center; }' +
      '}';

    var style = document.createElement('style');
    style.setAttribute('data-qf-plugin', 'confirmationStep');
    style.textContent = css;
    document.head.appendChild(style);
  }

  /**
   * Create the modal DOM and append it to the document body.
   */
  function createModal(fields, fieldValues) {
    injectStyles();

    var summaryRows = buildSummaryRows(fields, fieldValues);

    // Build modal HTML
    var html =
      '<div class="qf-cs-modal" role="dialog" aria-modal="true" aria-labelledby="qf-cs-title">' +
        /* Focus trap start sentinel */
        '<span class="qf-cs-sr-only" tabindex="0" data-qf-cs-trap="start"></span>' +

        /* Header */
        '<div class="qf-cs-header">' +
          '<span class="qf-cs-header-title" id="qf-cs-title">Review Your Submission</span>' +
          '<button type="button" class="qf-cs-close" aria-label="Close review" data-qf-cs-action="close">&times;</button>' +
        '</div>' +

        /* Body */
        '<div class="qf-cs-body">' +
          '<table class="qf-cs-table"><tbody>' +
            summaryRows +
          '</tbody></table>' +
        '</div>' +

        /* Footer */
        '<div class="qf-cs-footer">' +
          '<button type="button" class="qf-cs-btn qf-cs-btn-edit" data-qf-cs-action="edit">Edit</button>' +
          '<button type="button" class="qf-cs-btn qf-cs-btn-confirm" data-qf-cs-action="confirm">Confirm &amp; Submit</button>' +
        '</div>' +

        /* Focus trap end sentinel */
        '<span class="qf-cs-sr-only" tabindex="0" data-qf-cs-trap="end"></span>' +
      '</div>';

    var overlay = document.createElement('div');
    overlay.className = 'qf-cs-overlay';
    overlay.setAttribute('role', 'presentation');
    overlay.innerHTML = html;

    document.body.appendChild(overlay);
    modalOverlay = overlay;

    // Animate in
    requestAnimationFrame(function () {
      overlay.classList.add('qf-cs-visible');
    });

    // Wire up event handlers
    bindModalEvents(overlay);

    // Move focus to the confirm button
    var confirmBtn = overlay.querySelector('[data-qf-cs-action="confirm"]');
    if (confirmBtn) confirmBtn.focus();
  }

  /**
   * Bind click, keyboard, and focus-trap events on the modal.
   */
  function bindModalEvents(overlay) {
    // Delegate clicks
    overlay.addEventListener('click', function (e) {
      var action = getAction(e.target);
      if (action === 'close' || action === 'edit') {
        closeModal();
      } else if (action === 'confirm') {
        confirmAndSubmit();
      }
      // Click on backdrop closes
      if (e.target === overlay) {
        closeModal();
      }
    });

    // Keyboard handling
    overlay.addEventListener('keydown', function (e) {
      // Escape closes
      if (e.key === 'Escape') {
        e.preventDefault();
        closeModal();
        return;
      }

      // Focus trap: if Tab reaches a sentinel, wrap around
      if (e.key === 'Tab') {
        var target = e.target;
        var trap = target.getAttribute('data-qf-cs-trap');
        if (trap === 'end' && !e.shiftKey) {
          e.preventDefault();
          var firstFocusable = overlay.querySelector('.qf-cs-close');
          if (firstFocusable) firstFocusable.focus();
        } else if (trap === 'start' && e.shiftKey) {
          e.preventDefault();
          var lastFocusable = overlay.querySelector('[data-qf-cs-action="confirm"]');
          if (lastFocusable) lastFocusable.focus();
        }
      }
    });
  }

  /**
   * Walk up from an element to find the closest data-qf-cs-action value.
   */
  function getAction(el) {
    while (el && el !== document.body) {
      var action = el.getAttribute('data-qf-cs-action');
      if (action) return action;
      el = el.parentElement;
    }
    return null;
  }

  /**
   * Close and remove the modal, returning focus to the submit button.
   */
  function closeModal() {
    if (!modalOverlay) return;

    modalOverlay.classList.remove('qf-cs-visible');

    // Wait for animation before removing
    setTimeout(function () {
      if (modalOverlay && modalOverlay.parentNode) {
        modalOverlay.parentNode.removeChild(modalOverlay);
      }
      modalOverlay = null;
    }, 220);

    // Return focus to the submit button
    var submitBtn = document.getElementById('qf-submit-btn') ||
                    document.querySelector('.qf-submit-btn');
    if (submitBtn) submitBtn.focus();
  }

  /**
   * User confirmed: set flag, close modal, and re-trigger submit.
   */
  function confirmAndSubmit() {
    reviewConfirmed = true;
    closeModal();

    // Re-trigger the form submit after modal teardown
    setTimeout(function () {
      var submitBtn = document.getElementById('qf-submit-btn') ||
                      document.querySelector('.qf-submit-btn');
      if (submitBtn) {
        submitBtn.click();
      }
    }, 250);
  }

  // ---------------------------------------------------------------------------
  // Plugin registration
  // ---------------------------------------------------------------------------

  QuikFormsPlugins.register('confirmationStep', {
    hooks: {
      /**
       * onFormLoad: Change the submit button text to "Review & Submit".
       */
      onFormLoad: function (context) {
        var submitBtn = document.getElementById('qf-submit-btn') ||
                        document.querySelector('.qf-submit-btn');
        if (submitBtn) {
          submitBtn.textContent = 'Review & Submit';
        }
      },

      /**
       * onValidate: Two-phase interception.
       *
       * Phase 1 (reviewConfirmed === false):
       *   Build the review modal, show it, and abort submission.
       *
       * Phase 2 (reviewConfirmed === true):
       *   Allow submission to proceed normally and reset the flag.
       */
      onValidate: function (context) {
        if (reviewConfirmed) {
          // Phase 2 - allow submission
          reviewConfirmed = false;
          return { abort: false };
        }

        // Phase 1 - show modal and abort
        var fields = (context.formData && context.formData.displayFields) || [];
        var fieldValues = context.fieldValues || {};

        createModal(fields, fieldValues);

        return { abort: true };
      }
    }
  });

})();

/**
 * Entry point called by the QuikForms plugin loader after the static resource
 * is loaded. The IIFE above already self-registers, so this is a no-op safety
 * net in case the loader invokes the entry point explicitly.
 *
 * @param {object} registry - The QuikFormsPlugins registry instance
 */
function initQuikFormsConfirmationStep(registry) {
  // Plugin was already registered by the IIFE above.
  // This function exists solely to satisfy the JS_Entry_Point__c contract.
}
